<?php

namespace ShortLink\Http\Controllers;

use Illuminate\Http\Request;
use DB;
use Auth;
use ShortLink\Models\ShortLinks;
use ShortLink\Models\ShortLinkSetting;
use Illuminate\Support\Str;
use ShortLink\Models\LinkVisitor;
use ShortLink\Models\UrlVisit;
use Illuminate\Support\Facades\Request as UserRequest;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
 
use App\Http\Controllers\Controller;

class ShortLinkController extends Controller
{
    public function create()
    {
        $settings = ShortLinkSetting::first();
        
        // Check if guest access is allowed
        if (!Auth::check() && !$settings->allow_guest_shortening) {
            return redirect()->route('login')->with('error', 'Please login to create short links.');
        }
        
        return view('shortlink::create', [
            'settings' => $settings,
            'isGuest' => !Auth::check()
        ]);
    }

    public function store(Request $request)
    {
        $settings = ShortLinkSetting::first();
        
        // Check if guest access is allowed
        if (!Auth::check() && !$settings->allow_guest_shortening) {
            return redirect()->route('login')->with('error', 'Please login to create short links.');
        }

        $data = $request->validate([
            'long_url' => 'required|string',
            'slug' => 'nullable|string|alpha_dash|unique:short_links,backhalf',
            'expires_at' => 'nullable|date',
            'password' => 'nullable|string|min:3',
            'generate_qr' => 'nullable|boolean',
        ]);

        // If slug is not provided, generate a random one
        $slug = $data['slug'] ?? Str::random(Auth::check() ? 6 : 8);
        
        // If we're using a custom slug, it's already been validated as unique
        // If not, ensure the random slug is unique
        if (!isset($data['slug'])) {
            while (ShortLinks::where('backhalf', $slug)->exists()) {
                $slug = Str::random(Auth::check() ? 6 : 8);
            }
        }

        $domain = url('/go');
        $shortUrl = $domain . '/' . $slug;

        $model = new ShortLinks();
        $model->long_link = $data['long_url'];
        $model->domain = $domain;
        $model->short_link = $shortUrl;
        $model->backhalf = $slug;
        $model->title = $request->input('title', '');
        $model->user_id = Auth::check() ? Auth::id() : 0; // Set to 0 for guests
        $model->is_guest = !Auth::check(); // Mark as guest link
        $model->generate_qr = (bool) $request->boolean('generate_qr');
        
        // Set expiration for guest links (30 days from now if not specified)
        if (!empty($data['expires_at'])) {
            $model->expires_at = $data['expires_at'];
        } elseif (!Auth::check()) {
            $model->expires_at = now()->addDays(30);
        }
        
        // Handle password protection
        if (!empty($data['password'])) {
            $model->password = bcrypt($data['password']);
            $model->original_password = $data['password'];
        }
        
        $model->is_paused = false;
        $model->created_at = now();
        $model->save();

        // For guests, show a different success message with the full URL
        if (!Auth::check()) {
            return redirect()->back()
                ->with('success', 'Short link created successfully! Your short URL is: ' . $shortUrl)
                ->with('short_url', $shortUrl);
        }

        return redirect(url('crm/projects/short-links'))
            ->with('success', 'Short link created successfully!');
    }

    public function result($id)
    {
        $link = ShortLinks::findOrFail($id);
        $qr = null;
        // use public QR service for preview/download
        $qr = 'https://api.qrserver.com/v1/create-qr-code/?size=220x220&data=' . urlencode($link->short_link);
        return view('shortlink::result', ['link' => $link, 'qr' => $qr]);
    }
    public function shortLinkRedirect(Request $request)
    {
        $link = $request->link;

        $check = ShortLinks::where('backhalf', $link)->first();
        if (!empty($check)) {
            $longLink = $check->long_link;
            if (!preg_match('/^(http|https):\/\//', $longLink)) {
                $longLink = 'https://' . $longLink;
            }

            $this->recordVisit($check->id, $request);
            return redirect($longLink);
        }
    }

    public function goRedirect(Request $request, $slug)
    {
        $link = ShortLinks::where('backhalf', $slug)->first();
        if (!$link) {
            return response()->view('shortlink::protect', [
                'state' => 'invalid',
                'link' => null,
            ], 404);
        }

        // Paused
        if (!empty($link->is_paused)) {
            return response()->view('shortlink::protect', [
                'state' => 'paused',
                'link' => $link,
            ]);
        }

        // Expired
        if (!empty($link->expires_at) && now()->greaterThan($link->expires_at)) {
            return response()->view('shortlink::protect', [
                'state' => 'expired',
                'link' => $link,
            ]);
        }

        // Check for password protection
        $settings = ShortLinkSetting::getOrCreateDefault();
        if (!empty($link->password)) {
            if ($request->isMethod('post')) {
                $password = $request->input('password');
                if (password_verify($password, $link->password)) {
                    $this->recordVisit($link->id, $request);
                    return redirect()->away($this->normalizeUrl($link->long_link));
                }
                return response()->view('shortlink::protect', [
                    'state' => 'prompt',
                    'link' => $link,
                    'error' => 'Invalid password',
                ]);
            }
            return response()->view('shortlink::protect', [
                'state' => 'prompt',
                'link' => $link,
            ]);
        }

        $this->recordVisit($link->id, $request);
        return redirect()->away($this->normalizeUrl($link->long_link));
    }

    private function normalizeUrl($longLink)
    {
        if (!preg_match('/^(http|https):\/\//', $longLink)) {
            return 'https://' . $longLink;
        }
        return $longLink;
    }

    private function recordVisit($shortLinkId, Request $request): void
    {
        try {
            $ua = $request->userAgent();
            $ip = $request->ip();
            $referrer = $request->headers->get('referer');
            $device = $this->deviceFromUA($ua);
            
            // Get country from IP - try multiple methods
            $country = $this->getCountryFromIP($ip);
            
            // If we still don't have a country, try to get from Cloudflare header if available
            if (empty($country) || $country === 'Unknown') {
                $country = $request->header('CF-IPCountry');
            }
            
            // Final fallback
            $country = $country ?: 'Unknown';

            // Log the visit with all available information
            $visitData = [
                'short_link_id' => $shortLinkId,
                'ip' => $ip,
                'country' => $country,
                'device' => $device,
                'browser' => $this->getBrowserFromUA($ua),
                'referrer' => $referrer ?: 'Direct/None',
                'user_agent' => $ua,
                'created_at' => now(),
                'updated_at' => now(),
            ];
            
            UrlVisit::create($visitData);
            
        } catch (\Throwable $e) {
            \Illuminate\Support\Facades\Log::error('recordVisit failed: '.$e->getMessage());
        }
    }
    
    /**
     * Get country from IP using multiple fallback methods
     */
    private function getCountryFromIP($ip)
    {
        // Skip local IPs
        if (in_array($ip, ['127.0.0.1', '::1'])) {
            return 'Local';
        }
        
        // Method 1: Try geoip() helper if available
        if (function_exists('geoip')) {
            try {
                $geoip = geoip($ip);
                if (!empty($geoip['country'])) {
                    return $geoip['country'];
                }
            } catch (\Exception $e) {
                \Illuminate\Support\Facades\Log::warning('geoip() failed: ' . $e->getMessage());
            }
        }
        
        // Method 2: Try geoip_country_code_by_name
        if (function_exists('geoip_country_code_by_name')) {
            $countryCode = @geoip_country_code_by_name($ip);
            if ($countryCode) {
                return $countryCode;
            }
        }
        
        // Method 3: Try file_get_contents to ipinfo.io (free tier)
        try {
            $details = @file_get_contents("https://ipinfo.io/{$ip}/json");
            if ($details !== false) {
                $details = json_decode($details, true);
                if (isset($details['country'])) {
                    return $details['country'];
                }
            }
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::warning('ipinfo.io failed: ' . $e->getMessage());
        }
        
        return 'Unknown';
    }

    private function deviceFromUA(?string $ua): string
    {
        $ua = $ua ?? '';
        $u = strtolower($ua);
        if (strpos($u, 'tablet') !== false || strpos($u, 'ipad') !== false) return 'Tablet';
        if (strpos($u, 'mobile') !== false || strpos($u, 'iphone') !== false || strpos($u, 'android') !== false) return 'Mobile';
        if ($ua === '') return 'Unknown';
        return 'Desktop';
    }
    
    private function getBrowserFromUA(?string $ua): string
    {
        $ua = strtolower($ua ?? '');
        
        if (strpos($ua, 'msie') !== false || strpos($ua, 'trident') !== false) return 'Internet Explorer';
        if (strpos($ua, 'edg') !== false) return 'Microsoft Edge';
        if (strpos($ua, 'opr') !== false || strpos($ua, 'opera') !== false) return 'Opera';
        if (strpos($ua, 'chrome') !== false) return 'Chrome';
        if (strpos($ua, 'safari') !== false) return 'Safari';
        if (strpos($ua, 'firefox') !== false) return 'Firefox';
        
        return 'Unknown';
    }

    // Resource-style methods
    public function show($id)
    {
        $link = ShortLinks::findOrFail($id);
        return view('shortlink::result', ['link' => $link, 'qr' => 'https://api.qrserver.com/v1/create-qr-code/?size=220x220&data=' . urlencode($link->short_link)]);
    }

    public function stats($id)
    {
        $link = ShortLinks::findOrFail($id);
        $base = UrlVisit::query()->where('short_link_id', $link->id);
        $totalClicks = (clone $base)->count();
        $uniqueVisitors = (clone $base)->distinct()->count('ip');

        // Get top referrers
        $referrers = (clone $base)
            ->select(
                DB::raw("COALESCE(NULLIF(referrer,''),'Direct/None') as referrer"), 
                DB::raw('COUNT(*) as clicks')
            )
            ->groupBy('referrer')
            ->orderByDesc('clicks')
            ->limit(10)
            ->get();

        // Get countries data
        $countries = (clone $base)
            ->select(
                DB::raw("COALESCE(NULLIF(country,''),'Unknown') as country"), 
                DB::raw('COUNT(*) as clicks')
            )
            ->whereNotNull('country')
            ->where('country', '!=', '')
            ->groupBy('country')
            ->orderByDesc('clicks')
            ->get();

        // Get devices data
        $devices = (clone $base)
            ->select(
                DB::raw("COALESCE(NULLIF(device,''),'Unknown') as device"), 
                DB::raw('COUNT(*) as clicks')
            )
            ->groupBy('device')
            ->orderByDesc('clicks')
            ->get();

        // Get browsers data
        $browsers = (clone $base)
            ->select(
                DB::raw("COALESCE(NULLIF(browser,''),'Unknown') as browser"), 
                DB::raw('COUNT(*) as clicks')
            )
            ->whereNotNull('browser')
            ->where('browser', '!=', '')
            ->groupBy('browser')
            ->orderByDesc('clicks')
            ->get();

        // Get daily clicks for the last 30 days
        $daily = (clone $base)
            ->where('created_at', '>=', now()->subDays(29))
            ->select(
                DB::raw('DATE(created_at) as day'), 
                DB::raw('COUNT(*) as clicks')
            )
            ->groupBy('day')
            ->orderBy('day')
            ->get();

        // Ensure we have data for all days in the last 30 days
        $dateRange = collect();
        for ($i = 29; $i >= 0; $i--) {
            $date = now()->subDays($i)->format('Y-m-d');
            $dateRange->push((object)['day' => $date, 'clicks' => 0]);
        }

        // Merge with actual data
        $daily = $dateRange->map(function ($date) use ($daily) {
            $found = $daily->firstWhere('day', $date->day);
            return $found ?: (object)['day' => $date->day, 'clicks' => 0];
        });

        return view('shortlink::stats', [
            'link' => $link,
            'totalClicks' => $totalClicks,
            'uniqueVisitors' => $uniqueVisitors,
            'referrers' => $referrers,
            'countries' => $countries,
            'devices' => $devices,
            'browsers' => $browsers,
            'daily' => $daily,
        ]);
    }

    public function edit($id)
    {
        $link = ShortLinks::findOrFail($id);
        return view('shortlink::create', compact('link'));
    }

    public function update(Request $request, $id)
    {
        $link = ShortLinks::findOrFail($id);
        $data = $request->validate([
            'long_url' => 'required|string',
            'slug' => 'nullable|string|alpha_dash',
            'expires_at' => 'nullable|date',
            'password' => 'nullable|string|min:3',
            'password_off' => 'nullable|boolean',
        ]);

        // slug
        $slug = $data['slug'] ?? $link->backhalf;
        if (ShortLinks::where('backhalf', $slug)->where('id', '!=', $link->id)->exists()) {
            return back()->withErrors(['slug' => 'Slug already taken'])->withInput();
        }

        $link->long_link = $data['long_url'];
        $link->backhalf = $slug;
        $link->short_link = url('/go') . '/' . $slug;
        $link->expires_at = $data['expires_at'] ?? null;
        $link->generate_qr = (bool) $request->boolean('generate_qr');
        if (!empty($data['password_off'])) {
            $link->password = null;
            $link->original_password = null;
        } elseif (!empty($data['password'])) {
            $link->password = bcrypt($data['password']);
            $link->original_password = $data['password'];
        }
        $link->save();

        return redirect(url('crm/projects/short-links'));
    }

    public function destroy($id)
    {
        ShortLinks::where('id', $id)->delete();
        return redirect(url('crm/projects/short-links'));
    }

    public function pause($id)
    {
        ShortLinks::where('id',$id)->update(['is_paused' => true]);
        return back();
    }

    public function resume($id)
    {
        ShortLinks::where('id',$id)->update(['is_paused' => false]);
        return back();
    }

    public function deleteShortLink(Request $request)
    {
        ShortLinks::where('id', $request->id)->delete();
        return response()->json(['result'=>true]);
    }

    public function addShorterLink(Request $request)
    {
        $destination = $request->destination;
        $domain = $request->domain;
        $title = $request->title;
        $backhalf = $request->backhalf;
        $domain_type = $request->domain_type;
        $original_domain = $request->original_domain;

        if ($domain_type == 'current') {
            if (empty($backhalf)) {
                do {
                    $randomString = Str::random(6);
                    $exists = ShortLinks::where('backhalf', $randomString)->exists();
                } while ($exists);

                $backhalf = $randomString;
            }else{

                $checkBackhalf = ShortLinks::where('backhalf', $backhalf)->first();
                if(!empty($checkBackhalf))
                {
                    return response()->json([
                        'result'=>false,
                        'msg'=>'Backhalf already exists'
                    ]);
                }    
            }

            $shortLinkCreate = new ShortLinks();
            $shortLinkCreate->long_link = $destination;
            $shortLinkCreate->domain = $domain;
            $shortLinkCreate->short_link = $domain.'/'.$backhalf;
            $shortLinkCreate->backhalf = $backhalf;
            $shortLinkCreate->title = $title;
            $shortLinkCreate->user_id = Auth::user()->id;
            $shortLinkCreate->created_at = now();
            $shortLinkCreate->save();

            return response()->json(['result'=>true]);
        }else{

            $domainUrl = $this->getBaseUrl($domain);
            if(!$domainUrl)
            {
                return response()->json(['result'=>false, 'msg'=>'Domain not found']);
            }

            $getProject = DB::table('project_crm')->where('web_url', $domainUrl)->first();
            if(empty($getProject))
            {
                return response()->json(['result'=>false, 'msg'=>'Crm not found']);
            }

            $project_id = $getProject->id;
            $key = $getProject->api_key;
            $url = $getProject->web_url;

            $data = [
                'title'=>$title,
                'domain'=>$domain,
                'backhalf'=>$backhalf,
                'destination'=>$destination
            ];

            $apiUrl = $url.'/api/add_short_link';
            try {
               
                $response = Http::withHeaders([
                    'api_key' => $key,
                ])->post($apiUrl, $data);

                if ($response->successful()) {
                    $json = $response->body();
                    $array = json_decode($json, true);
                    return response()->json($array);
                }

                return response()->json(['result'=>false, 'msg'=>'Some error occured']);
                
            } catch (\Exception $e) {
                return response()->json(['result'=>false, 'msg'=>$e->getMessage()]);
            }


        }
       
    }

    public function getBaseUrl($url)
    {
        $parsedUrl = parse_url($url);
        if (isset($parsedUrl['scheme']) && isset($parsedUrl['host'])) {
            // Build the base URL
            $baseUrl = $parsedUrl['scheme'] . '://' . $parsedUrl['host'];
            if (isset($parsedUrl['port'])) {
                $baseUrl .= ':' . $parsedUrl['port'];
            }
            return $baseUrl;
        }
        return null;
    }

    public function shortLinks(Request $request)
    {
        $perPage = $request->input('per_page', 10); // Default to 10 items per page
        
        // Check if the current user is an admin (role 2)
        $isAdmin = Auth::check() && Auth::user()->role == 2;
        
        // Base query
        $query = ShortLinks::query();
        
        // For non-admin users, only show their own links and guest links
        if (!$isAdmin) {
            $query->where(function($q) {
                $q->where('user_id', Auth::id())
                  ->orWhere('is_guest', true);
            });
        }
        
        // Order and paginate
        $links = $query->orderByDesc('id')->paginate($perPage);
        
        $data['links'] = $links;
        $data['external_links'] = [];
        
        // Get the base URL for the current request
        $baseUrl = url('/');
        
        // Add external links for non-admin users
        if (Auth::check() && !$isAdmin) {
            $externalLinks = [];
            // Add your external links here
            // Example:
            // $externalLinks[] = [
            //     'id' => 'ext1',
            //     'title' => 'Example External Link',
            //     'url' => 'https://example.com',
            //     'short_url' => $baseUrl . '/go/ext1',
            //     'clicks' => 0,
            //     'created_at' => now(),
            //     'is_external' => true
            // ];
            $data['external_links'] = $externalLinks;
        }
        
        return view('shortlink::short_links_page', $data);
    }

    public function dashboard()
    {
        $userId = Auth::user()->id;
        $totalLinks = ShortLinks::where('user_id', $userId)->count();
        $protectedLinks = ShortLinks::where('user_id', $userId)
            ->when(\Schema::hasColumn('short_links','password'), function($q){
                $q->whereNotNull('password')->where('password','!=','');
            }, function($q){ return $q->whereRaw('1=0'); })
            ->count();
        $ids = ShortLinks::where('user_id', $userId)->pluck('id');
        $totalClicks = $ids->isEmpty() ? 0 : LinkVisitor::whereIn('short_link_id', $ids)->count();

        return view('shortlink::dashboard', [
            'totalLinks' => $totalLinks,
            'protectedLinks' => $protectedLinks,
            'totalClicks' => $totalClicks,
        ]);
    }

    // Settings page (uses Eloquent model ShortLinkSetting)
    public function settings()
    {
        $settings = ShortLinkSetting::getOrCreateDefault();
        return view('shortlink::settings', ['settings' => $settings]);
    }

    public function saveSettings(Request $request)
    {
        $settings = ShortLinkSetting::getOrCreateDefault();
        $settings->id = 1;
        $settings->allow_guest_shortening = $request->boolean('allow_guest_shortening');
        $settings->allow_password_protection = $request->boolean('allow_password_protection');
        $settings->blocklisted_domains = trim((string)$request->input('blocklisted_domains', ''));
        $settings->save();
        return redirect()->route('shortlinks.settings');
    }

    public function index()
    {
        $data = [];
        $projects = DB::table('projects')->where('uid', Auth::user()->id)->get();
        $publishedProjects = [];
        if(!$projects->isEmpty())
        {
            foreach ($projects as $project) {
                $checkPublished = DB::table('project_crm')->where('project_id', $project->id)->first();
                if(!empty($checkPublished))
                {
                    $publishedProjects[] = ['id'=>$project->id, 'url'=>$checkPublished->web_url.'/s'];
                }
            }
        }
        $data['published'] = $publishedProjects;
        return view('shortlink::short_link', $data);
    }

    public function deleteShortLinkViaApi(Request $request)
    {
        $link = $request->link;
        $id = $request->id;
        $domainUrl = $this->getBaseUrl($link);
        
        $getProject = DB::table('project_crm')->where('web_url', $domainUrl)->first();
        if(empty($getProject))
        {
            return response()->json(['result'=>false, 'msg'=>'Crm not found']);
        }

        $project_id = $getProject->id;
        $key = $getProject->api_key;
        $url = $getProject->web_url;

        $apiUrl = $url.'/api/delete_short_link';
        try {
            $param = ['id' => $id];
            $response = Http::withHeaders([
                'api_key' => $key,
            ])->post($apiUrl, $param);

            if ($response->successful()) {
                $json = $response->body();
                $array = json_decode($json, true);
                return response()->json($array);
            }

            return response()->json(['result'=>false, 'msg'=>'Some error occured']);
            
        } catch (\Exception $e) {
            return response()->json(['result'=>false, 'msg'=>$e->getMessage()]);
        }

    }
}
